/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.services.resources.admin;

import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.NotAcceptableException;
import jakarta.ws.rs.NotFoundException;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.UriInfo;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.util.Calendar;
import java.util.Set;
import java.util.stream.Collectors;
import org.eclipse.microprofile.openapi.annotations.Operation;
import org.eclipse.microprofile.openapi.annotations.extensions.Extension;
import org.eclipse.microprofile.openapi.annotations.parameters.Parameter;
import org.eclipse.microprofile.openapi.annotations.tags.Tag;
import org.jboss.resteasy.reactive.NoCache;
import org.keycloak.common.crypto.CryptoIntegration;
import org.keycloak.common.util.KeystoreUtil;
import org.keycloak.common.util.PemUtils;
import org.keycloak.events.admin.OperationType;
import org.keycloak.events.admin.ResourceType;
import org.keycloak.models.ClientModel;
import org.keycloak.models.KeyManager;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.RealmModel;
import org.keycloak.models.utils.KeycloakModelUtils;
import org.keycloak.representations.KeyStoreConfig;
import org.keycloak.representations.idm.CertificateRepresentation;
import org.keycloak.services.ErrorResponseException;
import org.keycloak.services.resources.admin.AdminEventBuilder;
import org.keycloak.services.resources.admin.fgap.AdminPermissionEvaluator;
import org.keycloak.services.util.CertificateInfoHelper;

@Extension(name="x-smallrye-profile-admin", value="")
public class ClientAttributeCertificateResource {
    protected final RealmModel realm;
    private final AdminPermissionEvaluator auth;
    protected final ClientModel client;
    protected final KeycloakSession session;
    protected final AdminEventBuilder adminEvent;
    protected final String attributePrefix;

    public ClientAttributeCertificateResource(AdminPermissionEvaluator auth, ClientModel client, KeycloakSession session, String attributePrefix, AdminEventBuilder adminEvent) {
        this.realm = session.getContext().getRealm();
        this.auth = auth;
        this.client = client;
        this.session = session;
        this.attributePrefix = attributePrefix;
        this.adminEvent = adminEvent.resource(ResourceType.CLIENT);
    }

    @GET
    @NoCache
    @Produces(value={"application/json"})
    @Tag(name="Client Attribute Certificate")
    @Operation(summary="Get key info")
    public CertificateRepresentation getKeyInfo() {
        this.auth.clients().requireView(this.client);
        CertificateRepresentation info = CertificateInfoHelper.getCertificateFromClient(this.client, this.attributePrefix);
        return info;
    }

    @POST
    @NoCache
    @Path(value="generate")
    @Produces(value={"application/json"})
    @Tag(name="Client Attribute Certificate")
    @Operation(summary="Generate a new certificate with new key pair")
    public CertificateRepresentation generate() {
        this.auth.clients().requireConfigure(this.client);
        CertificateRepresentation info = KeycloakModelUtils.generateKeyPairCertificate((String)this.client.getClientId());
        CertificateInfoHelper.updateClientModelCertificateInfo(this.client, info, this.attributePrefix);
        this.adminEvent.operation(OperationType.ACTION).resourcePath((UriInfo)this.session.getContext().getUri()).representation(info).success();
        return info;
    }

    @POST
    @Path(value="upload")
    @Consumes(value={"multipart/form-data"})
    @Produces(value={"application/json"})
    @Tag(name="Client Attribute Certificate")
    @Operation(summary="Upload certificate and eventually private key")
    public CertificateRepresentation uploadJks() throws IOException {
        this.auth.clients().requireConfigure(this.client);
        try {
            CertificateRepresentation info = CertificateInfoHelper.getCertificateFromRequest(this.session);
            this.updateCertFromRequest(info);
            return info;
        }
        catch (IllegalStateException ise) {
            throw new ErrorResponseException("certificate-not-found", "Certificate or key with given alias not found in the keystore", Response.Status.BAD_REQUEST);
        }
    }

    @POST
    @Path(value="upload-certificate")
    @Consumes(value={"multipart/form-data"})
    @Produces(value={"application/json"})
    @Tag(name="Client Attribute Certificate")
    @Operation(summary="Upload only certificate, not private key")
    public CertificateRepresentation uploadJksCertificate() throws IOException {
        this.auth.clients().requireManage(this.client);
        try {
            CertificateRepresentation info = CertificateInfoHelper.getCertificateFromRequest(this.session);
            this.updateCertFromRequest(info);
            return info;
        }
        catch (IllegalStateException ise) {
            throw new ErrorResponseException("certificate-not-found", "Certificate or key with given alias not found in the keystore", Response.Status.BAD_REQUEST);
        }
    }

    private void updateCertFromRequest(CertificateRepresentation info) {
        if ("openid-connect".equals(this.client.getProtocol()) && info.getJwks() != null) {
            CertificateInfoHelper.updateClientModelJwksString(this.client, this.attributePrefix, info.getJwks());
        } else {
            CertificateInfoHelper.updateClientModelCertificateInfo(this.client, info, this.attributePrefix);
        }
        this.adminEvent.operation(OperationType.ACTION).resourcePath((UriInfo)this.session.getContext().getUri()).representation(info).success();
    }

    @POST
    @NoCache
    @Path(value="/download")
    @Produces(value={"application/octet-stream"})
    @Consumes(value={"application/json"})
    @Tag(name="Client Attribute Certificate")
    @Operation(summary="Get a keystore file for the client, containing private key and public certificate")
    public byte[] getKeystore(@Parameter(description="Keystore configuration as JSON") KeyStoreConfig config) {
        this.auth.clients().requireView(this.client);
        this.checkKeystoreFormat(config);
        CertificateRepresentation info = CertificateInfoHelper.getCertificateFromClient(this.client, this.attributePrefix);
        String privatePem = info.getPrivateKey();
        String certPem = info.getCertificate();
        if (privatePem == null && certPem == null) {
            throw new NotFoundException("keypair not generated for client");
        }
        if (privatePem != null && config.getKeyPassword() == null) {
            throw new ErrorResponseException("password-missing", "Need to specify a key password for jks download", Response.Status.BAD_REQUEST);
        }
        if (config.getStorePassword() == null) {
            throw new ErrorResponseException("password-missing", "Need to specify a store password for jks download", Response.Status.BAD_REQUEST);
        }
        byte[] rtn = this.getKeystore(config, privatePem, certPem);
        return rtn;
    }

    @POST
    @NoCache
    @Path(value="/generate-and-download")
    @Produces(value={"application/octet-stream"})
    @Consumes(value={"application/json"})
    @Tag(name="Client Attribute Certificate")
    @Operation(summary="Generate a new keypair and certificate, and get the private key file\n\nGenerates a keypair and certificate and serves the private key in a specified keystore format.\nOnly generated public certificate is saved in Keycloak DB - the private key is not.")
    public byte[] generateAndGetKeystore(@Parameter(description="Keystore configuration as JSON") KeyStoreConfig config) {
        this.auth.clients().requireConfigure(this.client);
        this.checkKeystoreFormat(config);
        if (config.getKeyPassword() == null) {
            throw new ErrorResponseException("password-missing", "Need to specify a key password for jks generation and download", Response.Status.BAD_REQUEST);
        }
        if (config.getStorePassword() == null) {
            throw new ErrorResponseException("password-missing", "Need to specify a store password for jks generation and download", Response.Status.BAD_REQUEST);
        }
        int keySize = config.getKeySize() != null && config.getKeySize() > 0 ? config.getKeySize() : 4096;
        int validity = config.getValidity() != null && config.getValidity() > 0 ? config.getValidity() : 3;
        Calendar calendar = Calendar.getInstance();
        calendar.add(1, validity);
        CertificateRepresentation info = KeycloakModelUtils.generateKeyPairCertificate((String)this.client.getClientId(), (int)keySize, (Calendar)calendar);
        byte[] rtn = this.getKeystore(config, info.getPrivateKey(), info.getCertificate());
        info.setPrivateKey(null);
        CertificateInfoHelper.updateClientModelCertificateInfo(this.client, info, this.attributePrefix);
        this.adminEvent.operation(OperationType.ACTION).resourcePath((UriInfo)this.session.getContext().getUri()).representation(info).success();
        return rtn;
    }

    private byte[] getKeystore(KeyStoreConfig config, String privatePem, String certPem) {
        try {
            String format = config.getFormat();
            KeyStore keyStore = CryptoIntegration.getProvider().getKeyStore(KeystoreUtil.KeystoreFormat.valueOf((String)format));
            keyStore.load(null, null);
            String keyAlias = config.getKeyAlias();
            if (keyAlias == null) {
                keyAlias = this.client.getClientId();
            }
            if (privatePem != null) {
                PrivateKey privateKey = PemUtils.decodePrivateKey((String)privatePem);
                X509Certificate clientCert = PemUtils.decodeCertificate((String)certPem);
                Certificate[] chain = new Certificate[]{clientCert};
                keyStore.setKeyEntry(keyAlias, privateKey, config.getKeyPassword().trim().toCharArray(), chain);
            } else {
                X509Certificate clientCert = PemUtils.decodeCertificate((String)certPem);
                keyStore.setCertificateEntry(keyAlias, clientCert);
            }
            if (config.isRealmCertificate() == null || config.isRealmCertificate().booleanValue()) {
                KeyManager keys = this.session.keys();
                String kid = keys.getActiveRsaKey(this.realm).getKid();
                Certificate certificate = keys.getRsaCertificate(this.realm, kid);
                String certificateAlias = config.getRealmAlias();
                if (certificateAlias == null) {
                    certificateAlias = this.realm.getName();
                }
                keyStore.setCertificateEntry(certificateAlias, certificate);
            }
            ByteArrayOutputStream stream = new ByteArrayOutputStream();
            keyStore.store(stream, config.getStorePassword().trim().toCharArray());
            stream.flush();
            stream.close();
            byte[] rtn = stream.toByteArray();
            return rtn;
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private void checkKeystoreFormat(KeyStoreConfig config) throws NotAcceptableException {
        if (config.getFormat() != null) {
            Set supportedKeystoreFormats = CryptoIntegration.getProvider().getSupportedKeyStoreTypes().collect(Collectors.toSet());
            try {
                KeystoreUtil.KeystoreFormat format = Enum.valueOf(KeystoreUtil.KeystoreFormat.class, config.getFormat().toUpperCase());
                if (config.getFormat() != null && !supportedKeystoreFormats.contains(format)) {
                    throw new NotAcceptableException("Not supported keystore format. Supported keystore formats: " + String.valueOf(supportedKeystoreFormats));
                }
            }
            catch (IllegalArgumentException iae) {
                throw new NotAcceptableException("Not supported keystore format. Supported keystore formats: " + String.valueOf(supportedKeystoreFormats));
            }
        }
    }
}

